/*
 * Copyright (c) 2024 Oray Inc. All rights reserved.
 *
 * No Part of this file may be reproduced, stored
 * na retrieval system, or transmitted, in any form, or by any means,
 * electronic, mechanical, photocopying, recording, or otherwise,
 * without the prior consent of Oray Inc.
 *
 * @author wuwenze
 */
package com.oray.sunlogin.api;

import com.oray.sunlogin.api.dto.LocateHostRequest;
import com.oray.sunlogin.api.dto.RemoteControlRequest;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;

/**
 * 向日葵主控端API
 */
public class SunloginControlClient extends SunloginClientApi {
    private static final Logger logger = LoggerFactory.getLogger(SunloginControlClient.class);

    public SunloginControlClient(String host, SunloginAuthClient authClient) {
        super(host, authClient);
    }

    /**
     * 调起主控端程序
     */
    public void launchApp() {
        String osName = System.getProperty("os.name").toLowerCase();
        if (!osName.startsWith("windows")) {
            throw new IllegalStateException("launchApp only works on Windows, self: " + osName);
        }

        try {
            // 在Windows上使用cmd /c start来启动sl-remote://协议的应用
            Process process = Runtime.getRuntime().exec("cmd /c start sl-remote://");
            int exitCode = process.waitFor();
            if (exitCode != 0) {
                throw new IOException("launch app failed with exit code " + exitCode);
            }
        } catch (IOException | InterruptedException e) {
            throw new IllegalStateException("launch app fail", e);
        }
    }

    /**
     * 检测主控端程序是否存活
     * @see https://sunlogin.apifox.cn/api-151875839
     */
    public Boolean isAlive() {
        final SunloginClientApiResult result = super.sendRequest("GET", "/is_alive", null, null);
        if (result.getCode() != 200) {
            logger.error("is alive fail[status={}, code={}, message={}]",
                    result.getHttpStatus(), result.getCode(), result.getMessage());
            return false;
        }
        return true;
    }


    /**
     * 指定设备ID/识别码发起远控
     * @see https://sunlogin.apifox.cn/api-150555663
     */
    public Boolean remoteControl(final RemoteControlRequest request) {
        final SunloginClientApiResult result = super.sendRequest("POST", "/remote_control", null, request.asMap());
        if (result.getCode() != 200) {
            logger.error("remote control fail[status={}, code={}, message={}]",
                    result.getHttpStatus(), result.getCode(), result.getMessage());
            return false;
        }
        return true;
    }

    /**
     * LocateHost 在设备列表中定位指定设备ID并发起远控
     * @see https://sunlogin.apifox.cn/api-152488517
     */
    public Boolean locateHost(final LocateHostRequest request) {
        final SunloginClientApiResult result = super.sendRequest("POST", "/locate_host", null, request.asMap());
        if (result.getCode() != 200) {
            logger.error("remote control fail[status={}, code={}, message={}]",
                    result.getHttpStatus(), result.getCode(), result.getMessage());
            return false;
        }
        return true;
    }
}