/*
 * Copyright (c) 2024 Oray Inc. All rights reserved.
 *
 * No Part of this file may be reproduced, stored
 * na retrieval system, or transmitted, in any form, or by any means,
 * electronic, mechanical, photocopying, recording, or otherwise,
 * without the prior consent of Oray Inc.
 *
 * @author wuwenze
 */
package com.oray.sunlogin.api;

import java.io.IOException;
import java.net.URI;
import java.net.URLEncoder;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.nio.charset.StandardCharsets;
import java.security.GeneralSecurityException;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.util.Map;
import java.util.StringJoiner;

import javax.crypto.BadPaddingException;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.NoSuchPaddingException;

import com.fasterxml.jackson.core.JacksonException;
import com.oray.sunlogin.api.dto.GetAuthorizationCodeRequest;
import com.oray.sunlogin.api.entity.AuthCode;
import com.oray.sunlogin.api.factory.ObjectMapperFactory;

/**
 * 向日葵客户端API调用
 */
public class SunloginClientApi {
    private final String host; // 桌面端API主机
    private final SunloginAuthClient authClient; // 授权客户端

    public SunloginClientApi(final String host, final SunloginAuthClient authClient) {
        this.host = host;
        this.authClient = authClient;
    }

    public SunloginClientApiResult sendRequest(String method, String path, Map<String, String> queryParams,
            Map<String, String> fromData) {
        // 获取授权码
        final StringBuilder requestUrl = new StringBuilder(String.format("%s%s", this.host, path));
        final AuthCode authCode = this.authClient.getAuthorizationCode(new GetAuthorizationCodeRequest(method, path));
        try {
            // 构建业务请求，相关的参数需要加密处理
            final SunloginApiCipher cipher = new SunloginApiCipher(authCode.aesKey());
            final String encryptedQueryParams = this.encryptKvParams(queryParams, cipher);
            if (null != encryptedQueryParams && !encryptedQueryParams.isBlank()) {
                requestUrl.append("?").append(encryptedQueryParams);
            }

            final String encryptedFromData = this.encryptKvParams(fromData, cipher);
            final HttpRequest httpRequest = HttpRequest.newBuilder()//
                    .uri(URI.create(requestUrl.toString()))//
                    .header("Content-Type", "application/x-www-form-urlencoded")//
                    .header("Authorization", String.format("Bearer %s", authCode.code()))//
                    .method(method, HttpRequest.BodyPublishers.ofString(encryptedFromData)).build();

            // 发送请求
            final HttpResponse<String> httpResponse = LoggingHttpClient.newHttpClient()//
                    .sendBody(httpRequest, HttpResponse.BodyHandlers.ofString(), encryptedFromData);
            if (httpResponse.statusCode() != 200) {
                throw new IllegalStateException(
                        String.format("response not ok, [%d] %s", httpResponse.statusCode(), httpResponse.body()));
            }
            // 解析响应值
            final SunloginClientApiResult result = ObjectMapperFactory.getInstance()
                    .readValue(httpResponse.body(), SunloginClientApiResult.class);
            // Data字段需要解密处理
            if (result.getCode() == 200 && null != result.getData()) {
                result.setDecryptedData(cipher.decrypt(result.data));
            }
            result.setHttpStatus(httpResponse.statusCode());
            result.setHttpResponse(httpResponse.body());
            return result;
        } catch (NoSuchPaddingException | NoSuchAlgorithmException | InvalidKeyException | IllegalBlockSizeException
                | BadPaddingException | InterruptedException | IOException e) {
            if (e instanceof GeneralSecurityException) {
                throw new IllegalArgumentException("message crypto fail", e);
            }
            if (e instanceof JacksonException) {
                throw new IllegalArgumentException("unmarshal response body fail", e);
            }
            throw new IllegalStateException(String.format("send request(%s) fail", requestUrl), e);
        }
    }

    private String encryptKvParams(final Map<String, String> params, final SunloginApiCipher cipher)//
            throws NoSuchPaddingException, NoSuchAlgorithmException, InvalidKeyException, IllegalBlockSizeException,
            BadPaddingException {
        final StringJoiner result = new StringJoiner("&");
        if (null == params || params.isEmpty()) {
            return result.toString();
        }
        for (String k : params.keySet()) {
            final String v = params.get(k);
            if (null != v && !v.isEmpty()) {
                final String encrypted = cipher.encrypt(v);
                // 对加密后的 Base64 字符串进行 URL 编码
                final String urlEncoded = URLEncoder.encode(encrypted, StandardCharsets.UTF_8);
                result.add(String.format("%s=%s", k, urlEncoded));
            }
        }
        return result.toString();
    }

    public static class SunloginClientApiResult {
        private Integer code;
        private String message;
        private String data;
        private String decryptedData;
        private int httpStatus;
        private String httpResponse;

        public Integer getCode() {
            return code;
        }

        public void setCode(Integer code) {
            this.code = code;
        }

        public String getMessage() {
            return message;
        }

        public void setMessage(String message) {
            this.message = message;
        }

        public String getData() {
            return data;
        }

        public void setData(String data) {
            this.data = data;
        }

        public String getDecryptedData() {
            return decryptedData;
        }

        public void setDecryptedData(String decryptedData) {
            this.decryptedData = decryptedData;
        }

        public int getHttpStatus() {
            return httpStatus;
        }

        public void setHttpStatus(int httpStatus) {
            this.httpStatus = httpStatus;
        }

        public String getHttpResponse() {
            return httpResponse;
        }

        public void setHttpResponse(String httpResponse) {
            this.httpResponse = httpResponse;
        }
    }

}