/*
 * Copyright (c) 2024 Oray Inc. All rights reserved.
 *
 * No Part of this file may be reproduced, stored
 * na retrieval system, or transmitted, in any form, or by any means,
 * electronic, mechanical, photocopying, recording, or otherwise,
 * without the prior consent of Oray Inc.
 *
 * @author wuwenze
 */
package com.oray.sunlogin.api;

import com.oray.sunlogin.api.algorithm.Hmac;
import com.oray.sunlogin.api.algorithm.Md5;
import com.oray.sunlogin.api.dto.GetAuthorizationCodeRequest;
import com.oray.sunlogin.api.entity.AuthCode;
import com.oray.sunlogin.api.factory.ObjectMapperFactory;

import java.io.IOException;
import java.net.URI;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.time.Instant;
import java.util.UUID;


/**
 * 提供API集成，获取授权码的能力
 */
public class SunloginAuthClient {
    private final String host; // 授权服务地址
    private final String apiKey; // ApiKey，参见：【系统管理】-【API集成许可】
    private final String apiSecret; // ApiSecret, 参见：【系统管理】-【API集成许可】

    public SunloginAuthClient(String host, String apiKey, String apiSecret) {
        this.host = host;
        this.apiKey = apiKey;
        this.apiSecret = apiSecret;
    }

    /**
     * 获取授权码
     * @see https://sunlogin.apifox.cn/api-150317885
     */
    public AuthCode getAuthorizationCode(final GetAuthorizationCodeRequest request) {
        final String authUrl = String.format("%s/keyguard/authorization_code", this.host);
        try {
            final String requestBody = ObjectMapperFactory.getInstance().writeValueAsString(request);
            final HttpRequest httpRequest = this.createSignatureRequest("POST", authUrl, requestBody);

            final HttpResponse<String> httpResponse = LoggingHttpClient.newHttpClient()//
                    .sendBody(httpRequest, HttpResponse.BodyHandlers.ofString(), requestBody);
            if (httpResponse.statusCode() != 200) {
                throw new IllegalStateException(String.format("response not ok, [%d] %s", httpResponse.statusCode(), httpResponse.body()));
            }
            return ObjectMapperFactory.getInstance().readValue(httpResponse.body(), AuthCode.class);
        } catch (InterruptedException | IOException e) {
            throw new IllegalStateException("Get authorization code failed, authUrl=" + authUrl, e);
        }
    }

    private HttpRequest createSignatureRequest(String method, String url, String body) {
        // 准备签名所需参数
        final String contentMd5 = Md5.digest(body);
        final String timestamp = String.valueOf(Instant.now().getEpochSecond());
        final String nonce = UUID.randomUUID().toString();

        // 计算签名
        final String signature = Hmac.digestSHA256(//
                String.format("%s\n%s\n%s\n", contentMd5, timestamp, nonce), this.apiSecret);

        // 构建请求, 传入相关参数到Header
        return HttpRequest.newBuilder()//
                .uri(URI.create(url))//
                .header("Content-Md5", contentMd5)//
                .header("X-Ca-Api-Key", this.apiKey)//
                .header("X-Ca-Timestamp", timestamp)//
                .header("X-Ca-Nonce", nonce)//
                .header("X-Ca-Signature", signature)//
                .method(method, HttpRequest.BodyPublishers.ofString(body)).build();
    }

}