/*
 * Copyright (c) 2024 Oray Inc. All rights reserved.
 *
 * No Part of this file may be reproduced, stored
 * na retrieval system, or transmitted, in any form, or by any means,
 * electronic, mechanical, photocopying, recording, or otherwise,
 * without the prior consent of Oray Inc.
 *
 * @author wuwenze
 */
package com.oray.sunlogin.api;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLParameters;
import java.io.IOException;
import java.net.Authenticator;
import java.net.CookieHandler;
import java.net.ProxySelector;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.time.Duration;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;

import static java.net.http.HttpClient.Version.HTTP_1_1;

public class LoggingHttpClient extends HttpClient {
    private final static Logger logger = LoggerFactory.getLogger(LoggingHttpClient.class);

    private final HttpClient delegate;
    public LoggingHttpClient(final HttpClient httpClient) {
        this.delegate = httpClient;
    }

    public static LoggingHttpClient newHttpClient() {
        return new LoggingHttpClient(HttpClient.newBuilder().version(HTTP_1_1).build());
    }

    @Override
    public Optional<CookieHandler> cookieHandler() {
        return this.delegate.cookieHandler();
    }

    @Override
    public Optional<Duration> connectTimeout() {
        return this.delegate.connectTimeout();
    }

    @Override
    public Redirect followRedirects() {
        return this.delegate.followRedirects();
    }

    @Override
    public Optional<ProxySelector> proxy() {
        return this.delegate.proxy();
    }

    @Override
    public SSLContext sslContext() {
        return this.delegate.sslContext();
    }

    @Override
    public SSLParameters sslParameters() {
        return this.delegate.sslParameters();
    }

    @Override
    public Optional<Authenticator> authenticator() {
        return this.delegate.authenticator();
    }

    @Override
    public Version version() {
        return this.delegate.version();
    }

    @Override
    public Optional<Executor> executor() {
        return this.delegate.executor();
    }

    @Override
    public <T> HttpResponse<T> send(HttpRequest httpRequest, HttpResponse.BodyHandler<T> responseBodyHandler) throws IOException, InterruptedException {
        final HttpResponse<T> httpResponse = this.delegate.send(httpRequest, responseBodyHandler);
        logger.info("""
                Request: {} {}
                Request Headers: {}
                ----------------------------------------
                Response: {}
                Response Headers:  {}
                Response Body: {}""",
                httpRequest.method(), httpRequest.uri(), httpRequest.headers(),
                httpResponse.statusCode(), httpResponse.headers(), httpResponse.body());
        return httpResponse;
    }


    public <T> HttpResponse<T> sendBody(HttpRequest httpRequest, HttpResponse.BodyHandler<T> responseBodyHandler, String requestBody) throws IOException, InterruptedException {
        final HttpResponse<T> httpResponse = this.delegate.send(httpRequest, responseBodyHandler);
        logger.info("""
                Request: {} {}
                Request Headers: {}
                Request Body: {}
                ----------------------------------------
                Response: {}
                Response Headers:  {}
                Response Body: {}""",
                httpRequest.method(), httpRequest.uri(), httpRequest.headers(), requestBody,
                httpResponse.statusCode(), httpResponse.headers(), httpResponse.body());
        return httpResponse;
    }


    @Override
    public <T> CompletableFuture<HttpResponse<T>> sendAsync(HttpRequest request, HttpResponse.BodyHandler<T> responseBodyHandler) {
        return this.delegate.sendAsync(request, responseBodyHandler);
    }

    @Override
    public <T> CompletableFuture<HttpResponse<T>> sendAsync(HttpRequest request, HttpResponse.BodyHandler<T> responseBodyHandler, HttpResponse.PushPromiseHandler<T> pushPromiseHandler) {
        return this.delegate.sendAsync(request, responseBodyHandler);
    }
}